#ifndef _CMEMPOOL_CPP
#define _CMEMPOOL_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <Stdio.H>
#include <Stdlib.H>

#include "Debug.H"
#include "CMemPool.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CMemoryPool::IsDebug(void)
{
	#ifdef _DEBUG_MEMPOOL
		return true;
	#else
		return false;
	#endif
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	int CMemoryPool::GetMemAllocationCount(void)
	{
		return iUserAllocs;
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	int CMemoryPool::GetSlotAllocationCount(void)
	{
		return UserAlloc.iAllocated;
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	bool CMemoryPool::FreeAll(void)
	{
		int iSlot = 0;

		bool bResult = true;

		//Free all the memory stored in the AllocPool structure.
		while(iSlot < UserAlloc.iAllocated)
		{
			if(UserAlloc.AllocPool[iSlot].bFree == false)
			{
				if(!Free((void *)UserAlloc.AllocPool[iSlot].iAddress))
				{
					bResult = false;
				}
			}
			iSlot++;
		}

		return bResult;
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	int CMemoryPool::GetSlotByAddress(int iAddress)
	{
		EnterCriticalSection(&csMemLock);

		int iSlot = 0;

		while(iSlot < UserAlloc.iAllocated)
		{
			if(UserAlloc.AllocPool[iSlot].bFree == false)
			{
				if(UserAlloc.AllocPool[iSlot].iAddress == iAddress)
				{
					LeaveCriticalSection(&csMemLock);
					return iSlot;
				}
			}
			iSlot++;
		}

		char sText[1024];
		sprintf_s(sText, sizeof(sText), "Application attempted to access an unmanaged memory address at: %d.", iAddress);
		CMemoryPoolAssert("GetSlotByAddress", sText);

		__debugbreak();

		LeaveCriticalSection(&csMemLock);

		return -1; //Did not find the address specified.
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	int CMemoryPool::FindFreeAllocSlot(void)
	{
		EnterCriticalSection(&csMemLock);

		int iSlot = 0;

		while(iSlot < UserAlloc.iAllocated)
		{
			if(UserAlloc.AllocPool[iSlot].bFree)
			{
				LeaveCriticalSection(&csMemLock);
				return iSlot;
			}
			iSlot++;
		}

		int iMemory = (sizeof(ALLOCPOOL) * UserAlloc.iAllocated) + (sizeof(ALLOCPOOL) * UserAlloc.iAllocIncrement);

		if((UserAlloc.AllocPool = (ALLOCPOOL *) realloc(UserAlloc.AllocPool, iMemory)) == NULL)
		{
			char sText[1024];
			sprintf_s(sText, sizeof(sText), "Failed to reallocate the managed AllocPool structure for %d bytes.", iMemory);
			CMemoryPoolAssert("FindFreeAllocSlot", sText);

			__debugbreak();

			return -1;
		}

		UserAlloc.iAllocated = (UserAlloc.iAllocated + UserAlloc.iAllocIncrement);

		LeaveCriticalSection(&csMemLock);

		return FindFreeAllocSlot();
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	bool CMemoryPool::Init(void)
	{
		int iSlot = 0;

		InitializeCriticalSection(&csMemLock);
		
		memset(&UserAlloc, 0, sizeof(USERALLOC));
		memset(&UserAlloc.AllocPool, 0, sizeof(ALLOCPOOL));

		UserAlloc.iAllocated = 0;
		UserAlloc.iAllocIncrement = 100;

		if((UserAlloc.AllocPool = (ALLOCPOOL *) calloc(sizeof(ALLOCPOOL), UserAlloc.iAllocIncrement)) == NULL)
		{
			char sText[1024];
			sprintf_s(sText, sizeof(sText), "Memory pool Failed to initialize.");
			CMemoryPoolAssert("Init", sText);

			__debugbreak();

			return false;
		}

		UserAlloc.iAllocated = UserAlloc.iAllocIncrement;

		while(iSlot < UserAlloc.iAllocated)
		{
			UserAlloc.AllocPool[iSlot].bFree = true;
			UserAlloc.AllocPool[iSlot].iAddress = 0;
			UserAlloc.AllocPool[iSlot].iSize = 0;
			iSlot++;
		}

		return true;
	}
#else
	bool CMemoryPool::Init(void)
	{
		InitializeCriticalSection(&csMemLock);
		
		return true;
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	bool CMemoryPool::Destroy(void)
	{
		bool bResult = false;

		EnterCriticalSection(&csMemLock);

		bResult = FreeAll();

		free(UserAlloc.AllocPool);
			
		LeaveCriticalSection(&csMemLock);

		DeleteCriticalSection(&csMemLock);

		return bResult;
	}
#else
	void CMemoryPool::Destroy(void)
	{
		DeleteCriticalSection(&csMemLock);
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	bool CMemoryPool::Free(void *lpMemory)
	{
		EnterCriticalSection(&csMemLock);

		int iSlot = 0;

		if(lpMemory == NULL)
		{
			char sText[1024];
			sprintf_s(sText, sizeof(sText), "Application attempted to free a NULL pointer.");
			CMemoryPoolAssert("Free", sText);

			__debugbreak();

			LeaveCriticalSection(&csMemLock);
			return false;
		}

		if((iSlot = GetSlotByAddress((int)lpMemory)) >= 0)
		{
			//Free the memory that the user allocated, also mark the AllocPool slot as free.
			free(lpMemory);
			
			lpMemory = NULL;

			UserAlloc.AllocPool[iSlot].bFree = true;
			UserAlloc.AllocPool[iSlot].iAddress = 0;
			UserAlloc.AllocPool[iSlot].iSize = 0;

			iUserAllocs--;
		}
		else{
			char sText[1024];
			sprintf_s(sText, sizeof(sText), "Address [%d] not found in the managed memory pool.", lpMemory);
			CMemoryPoolAssert("Free", sText);

			__debugbreak();

			LeaveCriticalSection(&csMemLock);
			return false;
		}

		LeaveCriticalSection(&csMemLock);
		return true;
	}
#else
	void CMemoryPool::Free(void *lpMemory)
	{
		EnterCriticalSection(&csMemLock);

		free(lpMemory);

		LeaveCriticalSection(&csMemLock);
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	void *CMemoryPool::ReAllocate(void *lpMemory, const int iSize)
	{
		EnterCriticalSection(&csMemLock);

		int iSlot = 0;
		char *sMemory = NULL;

		if((iSlot = GetSlotByAddress((int)lpMemory)) >= 0)
		{
			if((sMemory = (char *) realloc(lpMemory, iSize)) == NULL)
			{
				char sText[1024];
				sprintf_s(sText, sizeof(sText), "Failed to reallocate managed user memory for %d bytes.", iSize);
				CMemoryPoolAssert("ReAllocate", sText);

				__debugbreak();

				LeaveCriticalSection(&csMemLock);
				return NULL;
			}

			UserAlloc.AllocPool[iSlot].bFree = false;
			UserAlloc.AllocPool[iSlot].iAddress = (int) sMemory;
			UserAlloc.AllocPool[iSlot].iSize = iSize;
		}
		else{
			char sText[1024];
			sprintf_s(sText, sizeof(sText), "Address [%d] not found in the managed memory pool.", lpMemory);
			CMemoryPoolAssert("ReAllocate", sText);

			//Address not found in the memory pool.
			__debugbreak();

			LeaveCriticalSection(&csMemLock);
			return NULL;
		}

		LeaveCriticalSection(&csMemLock);

		return sMemory;	
	}
#else
	void *CMemoryPool::ReAllocate(void *lpMemory, const int iSize)
	{

		EnterCriticalSection(&csMemLock);

		char *sMemory = (char *) realloc(lpMemory, iSize);

		LeaveCriticalSection(&csMemLock);

		return sMemory;
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG_MEMPOOL
	void *CMemoryPool::Allocate(const int iSize, const int iCount)
	{
		EnterCriticalSection(&csMemLock);

		char *sMemory = NULL;
		int iSlot = 0;

		if((sMemory = (char *) calloc(iSize, iCount)) == NULL)
		{
			char sText[1024];
			sprintf_s(sText, sizeof(sText), "Failed to allocate managed user memory for %d bytes.", iSize * iCount);
			CMemoryPoolAssert("Allocate", sText);

			__debugbreak();

			LeaveCriticalSection(&csMemLock);
			return NULL;
		}

		iSlot = FindFreeAllocSlot();

		UserAlloc.AllocPool[iSlot].iAddress = (int) sMemory;
		UserAlloc.AllocPool[iSlot].bFree = false;
		UserAlloc.AllocPool[iSlot].iSize = (iCount * iSize);

		iUserAllocs++;

		LeaveCriticalSection(&csMemLock);
		return sMemory;
	}
#else
	void *CMemoryPool::Allocate(const int iSize, const int iCount)
	{
		EnterCriticalSection(&csMemLock);

		char *sMemory = (char *) calloc(iCount, iSize);

		LeaveCriticalSection(&csMemLock);

		return sMemory;
	}
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif

